import os
import configparser
import pandas as pd
from datetime import datetime
import math as m
from geopy.distance import lonlat, distance


Config = configparser.ConfigParser()
"""
ConfigParser is a Python class which implements a basic configuration language for Python programs.
It provides a structure similar to Microsoft Windows INI files.
ConfigParser allows to write Python programs which can be customized by end users easily.
"""
Config.read('Ice_beacon_files.ini')
INPUT_DIRECTORY = Config.get('INPUT', 'INPUT_DIRECTORY')
PROCESSED_DIRECTORY = Config.get('INPUT', 'PROCESSED_DIRECTORY')
#  IGNORE_FILE_1 = Config.get('INPUT', 'IGNORE_FILE_1')
#  IGNORE_FILE_2 = Config.get('INPUT', 'IGNORE_FILE_2')
#  SPECIFIC_FILE = Config.get('INPUT', 'SPECIFIC_FILE')

"""
    open_files uses the parameters provided in the Ice_beacon_files.ini to access the data directory
    This function finds all CSV and Excel files and sends them to the appropriate helper functions(see functions
    clean_csv() and  clean_excel() for more information on helper functions.
    
    After each files has been processed by the appropriate helper function, the distance is calculated using the 
    function calc_distance() and the speed with calc_speed(). 
    
    Finally, each file is written out as a CSV file to the directory provided in Ice_beacon_files.ini
    
"""


def open_files():

    #  Open and clean Raw files from both Basys and Churchill folders. Convert all files to CSV
    for root, dirs, files in os.walk(INPUT_DIRECTORY):
        for filename in files:

            #  if filename != IGNORE_FILE_1 and filename != IGNORE_FILE_2:

            if filename.endswith(".csv"):
                csv_input_frame = pd.read_csv(os.path.join(root, filename))  # opens csv file
                cleaned_csv_file = clean_csv(csv_input_frame)
                cleaned_csv_file.to_csv(os.path.join(PROCESSED_DIRECTORY, filename), index=False)
                print("{} has been Processed".format(filename))

            elif filename.endswith(".xlsx"):  # and filename == SPECIFIC_FILE:

                excel_input_frame = pd.read_excel(os.path.join(root, filename), header=None)  # opens excel file
                clean_excel_file = clean_excel(excel_input_frame)

                new_name = filename.split(".")
                clean_excel_file.to_csv(os.path.join(PROCESSED_DIRECTORY, new_name[0] + ".csv"), index=False)

                print("{} has been Processed".format(filename))
                """
                excel_input_frame = pd.read_excel(os.path.join(root, filename), header=None)  # opens excel file
                clean_excel_file = clean_CT_file(excel_input_frame)

                new_name = filename.split(".")
                clean_excel_file.to_csv(os.path.join(PROCESSED_DIRECTORY, new_name[0] + ".csv"), index=False)

                print("{} has been Processed".format(filename))
                """

    for root, dirs, files in os.walk(PROCESSED_DIRECTORY):
        for filename in files:
            if filename.endswith(".csv"):
                ice_beacon_file = pd.read_csv(os.path.join(root, filename))  # opens csv file
                distance_calculation = calc_distance(ice_beacon_file)
                speed_calculation = calc_speed(distance_calculation)
                speed_calculation.to_csv(os.path.join(PROCESSED_DIRECTORY, filename), index=False)

                print("The distance & speed has been calculated for {}".format(filename))


"""
Clean_csv accepts a single parameter: The dataframe for the CSV file being processed & returns the processed dataframe 
to the function open_files()

This function converts the timestamp found in all input CSV files to match the format '%Y-%m-%d %H:%M:%S'
The column "message" contain both the "Battery Voltage" & "Internal Temperature" separated by a whitespace,
this is split and each value is put into a columns "Battery Voltage" & "Internal Temperature"

The following columns are renamed as follows:
    name - Beacon_ID
    lat - Latitude
    long - Longitude
    timestamp - Timestamp

The following columns are removed from the file:
    hdop
    pdop
    speed
    altitude
    heading
"""


def clean_csv(d_frame):
    battery_voltage = []
    internal_temperature = []
    timestamp = []
    drop_columns = ["id", "serial", "timestamp", "message", "hdop", "pdop", "speed", "altitude", "heading"]

    rows = len(d_frame["id"])
    for i in range(0, rows):
        splitTime = d_frame["timestamp"][i].split(" ")

        try:
            timestamp.append(datetime.strptime(splitTime[0] + " " + splitTime[1], '%Y-%m-%d %H:%M:%S'))
        except:
            timestamp.append(datetime.strptime(splitTime[0] + " " + splitTime[1], '%d/%m/%Y %H:%M:%S'))

        split_message = d_frame["message"][i].split(" ")
        battery_voltage.append(split_message[0])
        internal_temperature.append(split_message[1])

    for name in range(0, len(drop_columns)):
        d_frame.drop(drop_columns[name], axis=1, inplace=True)

    d_frame.rename(columns={"name": 'Beacon_ID'}, inplace=True)
    d_frame.rename(columns={"lat": 'Latitude'}, inplace=True)
    d_frame.rename(columns={"long": 'Longitude'}, inplace=True)

    d_frame.insert(3, "Timestamp", timestamp, True)
    d_frame.insert(4, "Battery Voltage", battery_voltage, True)
    d_frame.insert(5, "Internal Temperature", internal_temperature, True)

    return d_frame


"""
clean_excel accepts a single parameter: The dataframe for the CSV file being processed & returns the processed dataframe 
to the function open_files()

This function combines the columns 2-7 to create a timestamp of the match the format '%Y-%m-%d %H:%M:%S':

Column names added to each file:
    Beacon_ID
    Latitude
    Longitude
    Timestamp
    Battery Voltage
    Internal Temperature
"""


def clean_excel(d_frame):
    timestamp = []

    rows = len(d_frame[0])
    for i in range(0, rows):
        create_datetime = datetime(d_frame[3][i], d_frame[4][i], d_frame[5][i], d_frame[6][i], d_frame[7][i], d_frame[8][i])
        create_datetime = create_datetime.strftime("%Y-%m-%d %H:%M:%S")
        timestamp.append(create_datetime)

    for x in range(3, 9):
        d_frame.drop(x, axis=1, inplace=True)

    d_frame.insert(3, "Timestamp", timestamp, True)

    d_frame.rename(columns={0: 'Beacon_ID'}, inplace=True)
    d_frame.rename(columns={1: 'Latitude'}, inplace=True)
    d_frame.rename(columns={2: 'Longitude'}, inplace=True)
    d_frame.rename(columns={9: 'Battery Voltage'}, inplace=True)
    d_frame.rename(columns={10: 'Internal Temperature'}, inplace=True)

    return d_frame


"""
clean_CT_file accepts a single parameter: The dataframe for the CSV file being processed & returns the processed dataframe 
to the function open_files()

This function combines the columns 2-7 to create a timestamp of the match the format '%Y-%m-%d %H:%M:%S':

Column names added to each file:
    Beacon_ID
    Latitude
    Longitude
    Timestamp
    Battery Voltage
    Internal Temperature
"""

"""
def clean_CT_file(d_frame):
    beacon_id = []
    timestamp = []

    rows = len(d_frame[0])
    for i in range(0, rows):
        beacon_id.append("CT CEOS 300134010906880")
        create_datetime = datetime(d_frame[2][i], d_frame[3][i], d_frame[4][i], d_frame[5][i], d_frame[6][i], d_frame[7][i])
        create_datetime = create_datetime.strftime("%Y-%m-%d %H:%M:%S")
        timestamp.append(create_datetime)

    for x in range(2, 8):
        d_frame.drop(x, axis=1, inplace=True)

    d_frame.insert(0, "Beacon_ID", beacon_id, True)
    d_frame.insert(2, "Timestamp", timestamp, True)

    d_frame.rename(columns={0: 'Latitude'}, inplace=True)
    d_frame.rename(columns={1: 'Longitude'}, inplace=True)
    d_frame.rename(columns={8: 'Battery Voltage'}, inplace=True)
    d_frame.rename(columns={9: 'Internal Temperature'}, inplace=True)

    return d_frame
"""


"""
calc_distance accepts a single parameter: The dataframe for the CSV file being processed & returns the processed dataframe 
to the function open_files()

This function calculates the distance between every two points, in this case, the two points being each two rows of data
using the difference between the latitude and longitude of each point.
"""


def calc_distance(dframe):

    distance_list = [0]
    LONGITUDE_COL_NAME = "Longitude"
    LATITUDE_COL_NAME = "Latitude"
    row_count = len(dframe) - 1
    for i in range(0, row_count):
        geo1_no_altitude = dframe[LONGITUDE_COL_NAME][i], dframe[LATITUDE_COL_NAME][i]
        geo2_no_altitude = dframe[LONGITUDE_COL_NAME][i + 1], dframe[LATITUDE_COL_NAME][i + 1]
        dist_no_altitude = distance(lonlat(*geo2_no_altitude), lonlat(*geo1_no_altitude)).m
        distance_list.append(dist_no_altitude)

    dframe['distance_calculated_metres'] = pd.Series(distance_list)
    return dframe


"""
calc_speed accepts a single parameter: The dataframe for the CSV file being processed & returns the processed dataframe 
to the function open_files()

This function calculates the speed between every two points, in this case, the two points being each two rows of data
using the difference between the distance calculated in calc_distance() and timestamp.
"""


def calc_speed(dframe):
    speed_list = [0]
    TIMESTAMP = "Timestamp"
    row_count = len(dframe) - 1

    for i in range(0, row_count):
        time_1 = dframe[TIMESTAMP][i].split(" ")
        initial_time = int(datetime.strptime(dframe[TIMESTAMP][i].strip(" UTC"), '%Y-%m-%d %H:%M:%S').strftime("%s"))
        final_time = int(datetime.strptime(dframe[TIMESTAMP][i + 1].strip(" UTC"), '%Y-%m-%d %H:%M:%S').strftime("%s"))

        time_diff = final_time - initial_time

        if dframe['distance_calculated_metres'][i + 1] == 0:
            speed_list.append(0)
        else:
            speed = dframe['distance_calculated_metres'][i + 1] / time_diff
            speed_list.append(speed)

    dframe['speed_calculated_m/s'] = pd.Series(speed_list)

    return dframe


def main():
    open_files()


if __name__ == "__main__":
    start_time = datetime.now()
    main()
    end_time = datetime.now()
    run_time = end_time - start_time
    print(" \n\n\nTotal run time was {}".format(run_time))
    print("End of processing...")